#!/bin/sh -u

#  CollectDebugInfo.sh
#  BackupLoupe
#
#  Created by Robby Pählig on 28.12.2018.

function collectLogs()
{
    echo Collecting log files from ${log_path}… | tee -a "${log}"

    if test $(id -u) == 0; then
        log_path=/Library/Logs/$product
    fi

    if [[ ! -d ${log_path} ]]; then
        echo Log directory \"${log_path}\" does not exist. Skipping. | tee -a "${log}"
    else
        subdir=Logs
        mkdir -p "${outdir}/${subdir}"
        for f in ${log_path}/*; do
            if [[ "$f" == "${log_path}/*" ]]; then
                echo No files found. | tee -a "${log}"
                break
            fi
            echo Copying \"$f\"… | tee -a "${log}"
            cp "${f}" "${outdir}/${subdir}"
        done
    fi
}

function collectCrashReports()
{
    echo Collecting crash reports… | tee -a "${log}"

    subdir=Crash\ Reports
    mkdir -p "${outdir}/${subdir}"

    for f in $crashreport_path/*${product}*; do
        if [[ "$f" == "${crashreport_path}/*${product}*" ]]; then
            echo No files found. | tee -a "${log}"
            break
        fi
        echo Copying \"$f\"… | tee -a "${log}"
        cp "${f}" "${outdir}/${subdir}"
    done
}

function saveUserDefaults()
{
    echo Saving user defaults…
    defaults export ${identifier} $outdir/../${identifier}.plist
}

function usage ()
{
    echo "Usage: $(basename $0) IDENTIFIER OUTDIR USER (as root)"
	echo "       $(basename $0) IDENTIFIER OUTDIR"
}

echo Script was started at $(date) w/ uid $(id -u) and real uid $(id -ur): $0 $@

identifier=$1
if [[ -z "${identifier}" ]]; then
    usage
	exit 1
fi

product="${identifier##*.}"
if [[ ! ${identifier} == *.* ]]; then
    echo Invalid identifier: $identifier
    exit 1
fi

outdir=$2
if [[ -z "${outdir}" ]]; then
    usage
    exit 1
fi

log=$(dirname ${outdir})/collector.log

if [[ $(id -u) == 0 ]]; then
    user=$3
    if test -z "$3"; then
        echo No user specified >&2
        usage
        exit 1
    fi
    group=$(id -g ${user})
    if [[ $? -ne 0 ]]; then
        exit 1
    fi

    outdir="${outdir}/privileged"
    log_path=/Library/Logs/$product
    crashreport_path=/Library/Logs/DiagnosticReports
else
    outdir="${outdir}/unprivileged"
    log_path=~/Library/Logs/$product
    crashreport_path=~/Library/Logs/DiagnosticReports
fi

dir=$(dirname "${outdir}")
echo "Creating ${dir}" | tee -a "${log}"
mkdir -p "${outdir}"

collectLogs
collectCrashReports

if [[ $(id -u) == 0 ]]; then
    echo Creating ${dir}/launchctl-list-daemons.txt … | tee -a "${log}"
    /bin/launchctl list | /usr/bin/grep com.soma-zone > ${dir}/launchctl-list-daemons.txt

    echo Setting permissions… | tee -a "${log}"
    chown -R ${user}:${group} "${dir}"
    pushd "$(dirname "${dir}")" > /dev/null

    echo Creating archive… | tee -a "${log}"
    tar cf "${dir}.tar" "$(basename "${dir}")"

    echo Cleaning up… | tee -a "${log}"
    rm -r "$(basename "${dir}")"

    echo Compressing archive… | tee -a "${log}"
    bzip2 -9 "$(basename "${dir}").tar"

    echo Archive "$(basename "${dir}").tar.bz2" created.
    open -R "$(basename "${dir}").tar.bz2"
else
    echo Creating ${dir}/privileged-helpers.txt … | tee -a "${log}"
    /bin/ls -leadO@  /Library/PrivilegedHelperTools > "${dir}/privileged-helpers.txt"

    echo Creating ${dir}/launchctl-list-agents.txt … | tee -a "${log}"
    /bin/launchctl list | /usr/bin/grep com.soma-zone > ${dir}/launchctl-list-agents.txt

    echo Creating ${dir}/user-agents.txt … | tee -a "${log}"
    /bin/ls -leaO@ ~/Library/LaunchAgents > ${dir}/user-agents.txt

    echo Creating ${dir}/global-agents.txt … | tee -a "${log}"
    /bin/ls -leaO@ /Library/LaunchAgents > ${dir}/global-agents.txt

    echo Creating ${dir}/global-daemons.txt … | tee -a "${log}"
    /bin/ls -leaO@ /Library/LaunchDaemons > ${dir}/global-daemons.txt

    echo "Collecting Full Disk Access information…" | tee -a "${log}"
    echo "select client,client_type,auth_value,auth_reason,flags,last_modified from access where service = 'kTCCServiceSystemPolicyAllFiles'" | /usr/bin/sqlite3 /Library/Application\ Support/com.apple.TCC/TCC.db  > ${dir}/full-disk-access.txt

    saveUserDefaults
fi
